from typing import List, Dict, Optional
import sys
import os
import http.client
import json


def load_connections(path: str) -> Dict:
	"""
	Charge et analyse un fichier de log pour extraire les tentatives d'accès échouées par IP.

	@param path chemin vers le fichier de log à analyser
	@return dictionnaire associant chaque IP au nombre de tentatives échouées

	Postconditions : la fonction traite les exceptions et renvoie un dictionnaire vide dans le cas où :
	- le fichier n'existe pas ;
	- l'utilisateur n'a pas la permission de le lire ;
	- le fichier a un format incompatible.
	"""
	failures = {}
	try:
		with open(path, "rt") as f:
			while line := f.readline():
				parts = line[72:].split(" ")
				idx = 4 if parts[0] == "invalid" else 2
				ip = parts[idx]
				if ip not in failures: #si l'IP n'a pas encore été vue
					failures[ip] = 1
				else: #sinon, incrémenter le compteur
					failures[ip] += 1
	except FileNotFoundError:
		print("erreur : fichier non trouvé", file=sys.stderr)
	except PermissionError:
		print("erreur : droit de lecture nécessaire", file=sys.stderr)
	except IndexError:
		print("erreur : format de fichier incompatible", file=sys.stderr)
	except OSError as e:
		print("erreur : " + str(e), file=sys.stderr)
	return failures


def connections_statistics(data: Dict):
	"""
	Affiche des statistiques sur les tentatives d'accès échouées.

	@param data dictionnaire associant chaque IP au nombre de tentatives échouées
	"""
	nb = 0
	for val in data.values():
		nb += val
	print("nombre de tentatives : " + str(nb)) #environ 3 tentatives / minute (pare-feu)

	sorted_data = sort_dict(data)
	for ip, nb in sorted_data.items():
		print(ip + " : " + str(nb))

	country_data = sort_dict(load_countries(data))
	for country, nb in country_data.items():
		print(country + " : " + str(nb))


def sort_dict(data: Dict, reverse: bool = False) -> Dict:
	"""
	Trie un dictionnaire en fonctions des valeurs associées aux clés.

	@param data le dictionnaire à trier
	@param reverse True pour trier par ordre décroissant de valeurs
	@return nouveau dictionnaire trié par valeurs décroissantes

	Doctests:
	>>> sort_dict({"a": 3, "b": 1, "c": 2})
	{'a': 3, 'c': 2, 'b': 1}
	"""
	table = list(data.items()) #conversion du dictionnaire en liste
		#exemple : {"a": 3, "b": 1, "c": 2} → [('a', 3), ('b', 1), ('c', 2)]
	table.sort(key = lambda item: item[1], reverse = reverse) #tri de la liste
		#exemple d'item : ('a', 3) ; item[1] = 3 → c'est bien la valeur le critère de tri
	return dict(table) #conversion de la liste en dictionnaire


def load_countries(data: Dict) -> Dict:
	"""
	Géolocalise les adresses IP et compte le nombre de tentatives par pays.

	@param data dictionnaire associant chaque IP au nombre de tentatives échouées
	@return dictionnaire associant chaque pays au nombre total de tentatives

	Postconditions : en cas d'échec de la géolocalisation, le décompte est associé à "Unknown"
	"""
	country_data = {}
	for ip, nb in data.items():
		country = geolocalize_ip(ip)
		if country not in country_data: #si le pays est vu pour la première fois
			country_data[country] = nb
		else: #sinon, cumul du nombre de tentatives
			country_data[country] += nb
	return country_data


def geolocalize_ip(ip: str) -> str:
	"""
	Géolocalise une adresse IP pour obtenir son pays.

	@param ip adresse IP à géolocaliser
	@return nom du pays associé à l'IP, ou "Unknows" en cas d'échec

	Préconditions : connexion internet requise.
	"""
	country = "Unknown"
	conn = None
	try:
		# ~ conn = http.client.HTTPSConnection("ipapi.co")				#ipapi.co
		# ~ conn.request('GET', "/" + ip + "/country_name/")			#ipapi.co
		conn = http.client.HTTPConnection("ip-api.com")					#ip-api.com
		conn.request('GET', "/json/" + ip + "?fields=country")			#ip-api.com
		response = conn.getresponse()
		if response.status != 200: raise RuntimeError(f"HTTP {response.status}: {response.reason}")
		# ~ country = response.read().decode('utf-8')					#ipapi.co
		data = json.loads(response.read()) #désérialisation				#ip-api.com
		country = data['country']										#ip-api.com
	except ConnectionError as e:
		print("erreur : problème de connexion\n" + str(e), file=sys.stderr)
	except RuntimeError as e: #429 Too Many Requests
		print("erreur : status ≠ 200\n" + str(e), file=sys.stderr)
	except UnicodeDecodeError as e:
			print("erreur : réponse UTF-8 invalide\n" + str(e), file=sys.stderr)
	except json.JSONDecodeError as e:
		print("erreur : réponse JSON mal formée\n" + str(e), file=sys.stderr)
	finally:
		if conn is not None: conn.close()
	return country


if __name__ == "__main__":
	if len(sys.argv) != 2:
		print("erreur : argument manquant", file=sys.stderr)
		print("usage : " + sys.argv[0] + " chemin/vers/fichier.log", file=sys.stderr)
		sys.exit(1)
	data = load_connections(sys.argv[1])
	connections_statistics(data)
